﻿/*
VERSION: 	1.1
1.1		getWrappedCoords()		This function's non-looping behavior now allows diagonally sliding along the edges.

LIST OF FUNCTIONS: 
	generatePattern( original_pic )
	generateMoviePatternData( pattern_mc )
	makeRenderSurface( pattern_pic )
	makeMovieclipSurface( pattern_pic )
	modifyViewSize( renderData, {width:wid, height: hei} )
	applyYCrop( renderData, cropYScale )
	rotatePoint( input_p, apply_clockwise_radian )
	getWrappedCoords( pos_coords, wrapBounds, isLooping, dir_p )
	render( pattern_pic, pos_topLeft_p, radian, renderSurfaceData, zoomScale, eraseColor )
	
NOTE: 
	The position coords determine the top-left corner of the rendered image.
	If you want the position to appear to be centered,  you can use the "center_mat" provided by makeRenderSurface()  to offset your coordinates before sending them to getWrappedCoords()... or you can use your own arbitrary offset to change the apparent origin.
	
USAGE: 
	#include "functions/loopImageHelper.as"
	var BitmapData = flash.display.BitmapData;
	var Point = flash.geom.Point;
	var viewScale = 0.2;
	// get a world-map / rolling-image  (like a pokeball)
	var src_pic = BitmapData.loadBitmap( "test.png" );
	// make a loop-able pattern
	var patternData = loopImageHelper.generatePattern( src_pic );
	var pattern_pic = patternData.pattern_pic;
	var wrapCoords = patternData.wrapBounds;
	// keeps coords within the image
	var vel_p = new Point( 0, -10 );
	var coordsAfterVel_p = new Point( 500, 500 );
	var isLooping = true;
	var newCoords_p = loopImageHelper.getWrappedCoords( coordsAfterVel_p, wrapCoords, isLooping, vel_p );
	// prepare to render
	var surfaceData = loopImageHelper.makeRenderSurface( pattern_pic );
	// shrink the view-area
	surfaceData = loopImageHelper.modifyViewSize( surfaceData, {width:300, height: 300} );
	// apply cropYScale  (change the rotation-point of the output image, by cropping the render-image after rotation)
	var cropYScale = 0.6;
	surfaceData = loopImageHelper.applyYCrop( surfaceData, cropYScale );
	// render
	var angle = 0;
	var radian = angle * 2*Math.PI / 360;
	var rendered_pic = loopImageHelper.render( pattern_pic, newCoords_p, radian, surfaceData, viewScale );
	
	// display the render
	var test_mc = this.createEmptyMovieClip( "test_mc", 0 );
	test_mc._xscale = test_mc._yscale = 75;
	test_mc.attachBitmap( rendered_pic, 0 );
	// animate map-coords + display-angle
	onEnterFrame = function(){
		newCoords_p.y -= 10;		// change coords on the map
		radian += 0.01;				// spin the output image
		newCoords_p = loopImageHelper.getWrappedCoords( newCoords_p, wrapCoords, isLooping, vel_p );
		loopImageHelper.render( pattern_pic, newCoords_p, radian, surfaceData, viewScale );
	}// loop()
	

DETAILS ABOUT FUNCTIONS: 
	generatePattern( original_pic )
		generate pattern + wrap-coords
		input:	bitmap
		output	pattern_pic		(bitmap)
						wrapBounds		{ xMin,yMin,xMax,yMax }
	
	
	generateMoviePatternData( pattern_mc )
		movieClip pattern  =>  wrap-coords + pattern_mc ref
		input:	movieClip
		output	pattern_pic		(movieClip)		(named as "_pic" so that this data can seamlessly replace generatePattern() data)
						wrapBounds		{ xMin,yMin,xMax,yMax }
	
	
	makeRenderSurface( pattern_pic )
		prepare for rendering.  make an empty render pic  +  centering-data  +  cropping-data
		input:	pattern						(BitmapData)		(pattern bitmap)
		output:	render_pic				(BitmapData)		(output bitmap cropped to the maximum size that avoids visible clipping at 45-degree angles on a looping pattern)
						center_mat				(Matrix)				(coords stored as a matrix which controls the rotation-point)
						center_neg_mat		(Matrix)				(a matrix to undo whatever center_mat does)
						crop_mat					(Matrix)				(used to offset the rendered bitmap into the cropped area)
						clipRect					(Rectangle)			(used to crop the rendered bitmap)
	
	
	makeMovieclipSurface( pattern_pic )
		prepare for rendering.  make an empty render pic  +  centering-data  +  cropping-data
			pattern_movieClip  =>  empty render pic  +  centering-data  +  cropping-data
		input:	pattern						(MovieClip)			(pattern movieClip)
		output:	render_pic				(BitmapData)		(output bitmap scaled to the maximum size that avoids visible clipping at 45-degree angles on a looping pattern)
						center_mat				(Matrix)				(coords stored as a matrix which controls the rotation-point)
						center_neg_mat		(Matrix)				(a matrix to undo whatever center_mat does)
						crop_mat					(Matrix)				(used to offset the rendered bitmap into the cropped area)
						clipRect					(Rectangle)			(used to crop the rendered bitmap)
		
		The input movieClip should contain 4 identical images that are scaled down to 1024x1024 each (or smaller).  (Call generateMoviePatternData() instead of generatePattern() before using makeMovieclipSurface()
		The images themselves can be ANY resolution...
		... and this allows the displayed detail to be VERY high resolution.
		The movieClip/images can also be animated, which will automatically show up in the rendered output each time render() is called.
	
	
	modifyViewSize( renderData, newSize_rect )
		shrink the view-area
			Resizes the clipping-rectangle and rendering bitmap to the specified size
			renderSurfaceData + newSize_rectangle  =>  new renderSurfaceData with modified  clipRect  +  render_pic
		input:		object			(renderSurface data)
								clipRect
								render_pic
							Rectangle		(newSize_rect)
		output:		object			(new renderSurface data  (shallow-copy if input with modified  clipRect  +  render_pic)
	
	
	applyYCrop( renderData, yScale )
		apply cropYScale  (change the rotation-point of the output image, by cropping the render-image after rotation)
			Crop the clipping-rectangle and rendering bitmap down to the specified vertical-scale
			renderSurfaceData + yScalar  =>  new renderSurfaceData with modified  clipRect  +  render_pic
		input:		object			(renderSurface data)
								clipRect
								render_pic
							Rectangle		(newSize_rect)
		output:		object			(new renderSurface data  (shallow-copy if input with modified  clipRect  +  render_pic)
	
	
	
	rotatePoint( input_p, apply_clockwise_radian )
		Rotate a Point  (a Point object is technically a vector)
		vector + angle  =  rotated vector  (clockwise)
		
		input:	local_vector, clockwise_radian
		output:	rotated_vector
		
		This enables relative movement like so: 
			Take local velocity
			Apply global rotation to local velocity
			This results in a global velocity
			Add this global velocity to the global position
	
	
	getWrappedCoords( input_coords, wrapBounds, isLooping, dir_p )
		Wraps the specified coordinates to remain within the loop-able bounds of the image,  to maintain the illusion of an infinitely looping image
			coords  =>  wrapped-coords / edge-collided coords
		input:	x,y		(attempt these coords)
						xMin,yMin,xMax,yMax
						wrap-coords:  true/false			[optional]		
						x,y		(direction vector)			[optional]		(must be a Point object)		(if missing:  wrap = modulous,  no-wrap = naive edge-snap  (no sense of velocity)
		output:	Point:	x,y
		
		
	render( pattern_pic, pos_topLeft_p, radian, renderSurfaceData, zoomScale, eraseColor )
		Updates  renderSurfaceData.render_pic  with a rotated and scaled image based on pattern_pic
			coords + pattern  =>  output pic
		input:	bitmap			(pattern bitmap)
						x,y					(view-point top-left coords)
						radian			(view-point angle)
						renderSurfaceData
							render_pic				(BitmapData)		(output bitmap scaled to the maximum size that avoids visible clipping at 45-degree angles on a looping pattern)
							center_mat				(Matrix)				(coords stored as a matrix which controls the rotation-point)
							center_neg_mat		(Matrix)				(a matrix to undo whatever center_mat does)
							crop_mat					(Matrix)				(used to offset the rendered bitmap into the cropped area)
							clipRect					(Rectangle)			(used to crop the rendered bitmap)
		output:	bitmap			(rendered bitmap  AKA  renderSurfaceData.render_pic)
*/
var loopImageHelper = {
	/*****************************************
	*****************************************/
	
	
	
	/*****************************************
	generate pattern + wrap-coords
	input:	bitmap
	output	pattern_pic		(bitmap)
					wrapBounds		{ xMin,yMin,xMax,yMax }
	*****************************************/
	generatePattern: function( original_pic ){
		// convenient shortcuts
		var Point = flash.geom.Point;
		var BitmapData = flash.display.BitmapData;
		var Rectangle = flash.geom.Rectangle;
		
		// resolve passed parameters
		if( original_pic instanceof BitmapData === false )		throw new Error("generatePattern() requires BitmapData input");
		// internal parameters
		var edgeTransparent = false;
		
		
		
		// prepare the pattern for wrapping
		var ww = original_pic.width *2;
		var hh = original_pic.height *2;
		var pattern_pic = new BitmapData( ww,hh, edgeTransparent,0 );
		
		// copy the original into the prepped image
		var copy_rect = new Rectangle( 0,0, original_pic.width, original_pic.height );
		var paste_p = new Point( 0, 0 );
		pattern_pic.copyPixels( original_pic, copy_rect, paste_p );
		
		
		
		// add loop-copies to the prepped image
		// // create right buffer
		// // // copy left part
		var xx = 0;
		var yy = 0;
		var ww = original_pic.width;
		var hh = pattern_pic.height;
		var copy_rect = new Rectangle(xx,yy,ww,hh);
		// // // paste to the right
		var xx = original_pic.width;
		var yy = 0;
		var paste_p = new Point(xx,yy);
		pattern_pic.copyPixels( pattern_pic, copy_rect, paste_p );
		// // create bottom loop-copies
		// // // copy top half
		var xx = 0;
		var yy = 0;
		var ww = pattern_pic.width;
		var hh = original_pic.height;
		var copy_rect = new Rectangle(xx,yy,ww,hh);
		// // // paste to the bottom half
		var xx = 0;
		var yy = original_pic.height;
		var paste_p = new Point(xx,yy);
		pattern_pic.copyPixels( pattern_pic, copy_rect, paste_p );
		
		
		var wrapBounds = {
			xMin: 0,
			yMin: 0,
			xMax: original_pic.width,
			yMax: original_pic.height
		};
		/*	
		var halfWidth = math.floor(original_pic.width /2);
		var halfHeight = math.floor(original_pic.height /2);
		var wrapBounds = {
			xMin: halfWidth,
			yMin: halfHeight,
			xMax: original_pic.width +halfWidth,
			yMax: original_pic.height +halfHeight
		};
		*/
		
		var output = {
			pattern_pic: pattern_pic,
			wrapBounds: wrapBounds
		};
		output.wrapBounds.toString = function(){
			return "min: ("+this.xMin+", "+this.yMin+")  max: ("+this.xMax+", "+this.yMax+")";
		}
		return output;
	}, //generatePattern()
	
	
	
	/*****************************************
	movieClip pattern => wrap-coords + pattern_mc ref
	input:	movieClip
	output	pattern_pic		(movieClip)		(named as "_pic" so that this data can seamlessly replace generatePattern() data)
					wrapBounds		{ xMin,yMin,xMax,yMax }
	*****************************************/
	generateMoviePatternData: function( pattern_mc ){
		// resolve passed parameters
		var hasPattern = ( pattern_mc instanceof MovieClip );
		if( !hasPattern )		throw new Error( "makeMovieclipSurface() did not receive a pattern MovieClip" );
		
		var wrapBounds = {
			xMin: 0,
			yMin: 0,
			xMax: pattern_mc._width /2,
			yMax: pattern_mc._height /2
		};
		var output = {
			pattern_pic: pattern_mc,
			wrapBounds: wrapBounds
		};
		output.wrapBounds.toString = function(){
			return "min: ("+this.xMin+", "+this.yMin+")  max: ("+this.xMax+", "+this.yMax+")";
		}
		return output;
	}, // generateMoviePatternData()
	
	
	
	/*****************************************
	// Rotate a Point  (a Point object is technically a vector)
	vector + angle  =  rotated vector  (clockwise)
	
	input:	local_vector, clockwise_radian
	output:	rotated_vector
	
	This enables relative movement like so: 
		Take local velocity
		Apply global rotation to local velocity
		This results in a global velocity
		Add this global velocity to the global position
	*****************************************/
	rotatePoint: function( input_p, apply_clockwise_radian ){
		var Matrix = flash.geom.Matrix;
		var Point = flash.geom.Point;
		var work_mat = new Matrix( 1,0,0,1, -input_p.x, -input_p.y );
		work_mat.rotate( apply_clockwise_radian);
		var output = new Point( work_mat.tx, work_mat.ty );
		return output;
	}, // rotatePoint()
	
	
	
	/*****************************************
	wrap-coords + doWrap + coords + direction-vector = new coords
	input:	x,y		(attempt these coords)
					xMin,yMin,xMax,yMax
					wrap-coords:  true/false			[optional]		
					x,y		(direction vector)			[optional]		(must be a Point object)		(if missing:  wrap = modulous,  no-wrap = naive edge-snap  (no sense of velocity)
	output:	Point:	x,y
	*****************************************/
	getWrappedCoords: function( input_coords, wrapBounds, isLooping, dir_p ){
		// convenient shortcuts
		var Point = flash.geom.Point;
		
		// resolve passed parameters
		var hasCoords = !( input_coords.x === undefined  ||  input_coords.y === undefined );
		if( !hasCoords )		throw new Error("getWrappedCoords() did not receive coordinates");
		var hasNumCoords = !( isNaN(input_coords.x)  ||  isNaN(input_coords.y) );
		if( !hasNumCoords )		throw new Error("getWrappedCoords() received invalid non-numeric coordinates");
		var hasWrapBounds = !( wrapBounds.xMin === undefined || wrapBounds.yMin === undefined || wrapBounds.xMax === undefined || wrapBounds.yMax === undefined);
		if( !hasWrapBounds )		throw new Error("getWrappedCoords() did not receive wrapBounds");
		var hasNumWrapBounds = !( isNaN(wrapBounds.xMin) || isNaN(wrapBounds.yMin) || isNaN(wrapBounds.xMax) || isNaN(wrapBounds.yMax) );
		if( !hasNumWrapBounds )		throw new Error("getWrappedCoords() received invalid non-numeric wrapBounds");
		var hasLoopingParam = !( isLooping === undefined );
		if( !hasLoopingParam )		var isLooping = true;
		var hasDir = (dir_p instanceof Point);
		if( hasDir ){
			var dir_p = dir_p.clone();
			var unit_p = dir_p.clone();		unit_p.normalize(1);
		}
		
		
		// 
		var output = new Point( input_coords.x, input_coords.y );
		
		
		if( isLooping )
		{// if:  coordinates should wrap-around
			while( output.x < 0 )		output.x += wrapBounds.xMax;
			while( output.y < 0 )		output.y += wrapBounds.yMax;
			if( output.x >= wrapBounds.xMax )		output.x %= wrapBounds.xMax;
			if( output.y >= wrapBounds.yMax )		output.y %= wrapBounds.yMax;
		}// if:  coordinates should wrap-around
		
		else
		
		// This version allows for diagonally sliding along the edges.
		{// if:  coordinates should stop and collide at edges
			if( output.x < wrapBounds.xMin)		output.x = wrapBounds.xMin;
			else if( output.x > wrapBounds.xMax)		output.x = wrapBounds.xMax;
			if( output.y < wrapBounds.yMin)		output.y = wrapBounds.yMin;
			else if( output.y > wrapBounds.yMax)		output.y = wrapBounds.yMax;
		}// if:  coordinates should stop and collide at edges
		
		/*	
		// This version cancels out-of-bounds coordinates by back-tracking to the edge.  This does not allow for diagonally sliding along edges.  Movement simply stops dead at the edge.
		{// if:  coordinates should stop and collide at edges
			//  x is too small
			if( output.x < 0 ){
				if( hasDir  &&  unit_p.x !== 0 )
				{// if:   a direction vector is available
					// back-track along direction-vector to determine where you should be
					var xDiff = -output.x;		// get positive x
					output.y += (xDiff / unit_p.x) * unit_p.y;			//  xDiff / unit_p.x  determines how much the unit-vector needs to be scaled-up,  then it's applied to  unit_p.y
				}// if:   a direction vector is available
				output.x = 0;
			}// if:   x is too small
			
			// y is too small
			if( output.y < 0 ){
				if( hasDir  &&  unit_p.y !== 0  )
				{// if:   a direction vector is available
					// back-track along direction-vector to determine where you should be
					var yDiff = -output.y;		// get positive y
					output.x += (yDiff / unit_p.y) * unit_p.x;			//  yDiff / unit_p.y  determines how much the unit-vector needs to be scaled-up,  then it's applied to  unit_p.x
				}// if:   a direction vector is available
				output.y = 0;
			}// if:   y is too small
			
			// x is too big
			if( output.x >= wrapBounds.xMax ){
				if( hasDir  &&  unit_p.x !== 0  )
				{// if:   a direction vector is available
					// back-track along direction-vector to determine where you should be
					var xDiff = output.x - wrapBounds.xMax +1;
					output.y -= (xDiff / unit_p.x) * unit_p.y;			//  xDiff / unit_p.x  determines how much the unit-vector needs to be scaled-up,  then it's applied to  unit_p.y
				}// if:   a direction vector is available
				output.x = wrapBounds.xMax-1;
			}// if:  x is too big
			
			// y is too big
			if( output.y >= wrapBounds.yMax ){
				if( hasDir  &&  unit_p.y !== 0  )
				{// if:   a direction vector is available
					// back-track along direction-vector to determine where you should be
					var yDiff = output.y - wrapBounds.yMax +1;
					output.x -= (yDiff / unit_p.y) * unit_p.x;			//  yDiff / unit_p.y  determines how much the unit-vector needs to be scaled-up,  then it's applied to  unit_p.x
				}// if:   a direction vector is available
				output.y = wrapBounds.yMax-1;
			}// if:  x is too big
		}// if:  coordinates should stop and collide at edges
		*/
		
		
		return output;
	}, // getWrappedCoords()
	
	
	
	/*****************************************
	pattern_pic  =  empty output pic + centering-data
	input:	pattern						(BitmapData)		(pattern bitmap)
					isWrapping				(Boolean)				(Whether this pattern has a wrap-buffer,  or if it's just a single instance image)
	output:	render_pic				(BitmapData)		(output bitmap scaled to the maximum size that avoids visible clipping at 45-degree angles on a looping pattern)
					center_mat				(Matrix)				(coords stored as a matrix which controls the rotation-point)
					center_neg_mat		(Matrix)				(a matrix to undo whatever center_mat does)
					crop_mat					(Matrix)				(used to offset the rendered bitmap into the cropped area)
					clipRect					(Rectangle)			(used to crop the rendered bitmap)
	*****************************************/
	makeRenderSurface: function( pattern_pic, isWrapping ){
		var Matrix = flash.geom.Matrix;
		var BitmapData = flash.display.BitmapData;
		var Rectangle = flash.geom.Rectangle;
		
		// resolve passed parameters
		var hasPattern = ( pattern_pic instanceof BitmapData );
		if( !hasPattern )		throw new Error( "makeRenderSurface() did not receive a pattern BitmapData" );
		if( isWrapping === undefined )		isWrapping = true;
		
		// internal parameters
		var edgeOpaque = true;
		var halfImageScale = ( isWrapping )  ?  4  :  2;
		
		// Create the render-display image.
		// figure out the maximum safe crop area,  so that no out-of-image areas will ever be seen
		var aspectRatio = pattern_pic.width / pattern_pic.height;
		//var rad_45 = 0.78539816339744830961566084581988;		// 45 degrees, in radians  (where the rectangle touches the safe-circle of a square image)
		//var x_scalar = Math.cos(rad_45);
		//var y_scalar = Math.sin(rad_45);	// omitted because, at 45 degrees, sin() and cos() have the same value
		var x_scalar = 0.707106781186548;		// hard-code the resulting value because it will never change
		var y_scalar = x_scalar * aspectRatio;		// x_scalar is used because, at 45 degrees, sin() and cos() have the same value
		var crop_x = Math.round((pattern_pic.width/halfImageScale)		*(1-x_scalar));			// left offset	(half of a single image  (exclude duplicate buffer)
		var crop_y = Math.round((pattern_pic.height/halfImageScale)	*(1-y_scalar));			// top offset		(half of a single image  (exclude duplicate buffer)
		var crop_width = Math.round((pattern_pic.width/halfImageScale		*x_scalar) *2);
		var crop_height = Math.round((pattern_pic.height/halfImageScale	*y_scalar) *2);
		var crop_mat = new Matrix( 1,0,0,1, -crop_x, -crop_y );
		var crop_pic = new BitmapData(crop_width, crop_height, !edgeOpaque, 0);
		// crop_pic = new BitmapData( pattern_pic.width, pattern_pic.height, !edgeOpaque, 0);		// used to test clipRect  (without clipRect, a very large image is displayed)
		var clipRect = new Rectangle( 0,0, crop_width,crop_height  );
		
		var center_mat = 			new Matrix( 1,0,0,1, -(pattern_pic.width/halfImageScale), -(pattern_pic.height/halfImageScale) );		// half width  of a single image iteration  (exclude duplicate buffer on the right)
		var center_neg_mat = 	new Matrix( 1,0,0,1,  (pattern_pic.width/halfImageScale),  (pattern_pic.height/halfImageScale) );		// half height of a single image iteration  (exclude duplicate buffer on the bottom)
		
		var output = {
			render_pic: crop_pic, 
			center_mat: center_mat, 
			center_neg_mat: center_neg_mat, 
			crop_mat: crop_mat, 
			clipRect: clipRect
		}
		
		return output;
	}, // makeRenderSurface()
	
	
	
	/*****************************************
	pattern_movieClip  =  empty output pic + centering-data
	input:	pattern						(MovieClip)			(pattern movieClip)
	output:	render_pic				(BitmapData)		(output bitmap scaled to the maximum size that avoids visible clipping at 45-degree angles on a looping pattern)
					center_mat				(Matrix)				(coords stored as a matrix which controls the rotation-point)
					center_neg_mat		(Matrix)				(a matrix to undo whatever center_mat does)
					crop_mat					(Matrix)				(used to offset the rendered bitmap into the cropped area)
					clipRect					(Rectangle)			(used to crop the rendered bitmap)
	
	The input movieClip should contain 4 identical images that are scaled down to 1024x1024 each.  (Call generateMoviePatternData() instead of generatePattern()
	The images themselves can be ANY resolution.
	This allows the details to be VERY high resolution.
	The movieClip can also be animated, which will automatically show up in the rendered output each time redner() is called.
	*****************************************/
	makeMovieclipSurface: function( pattern_mc ){
		var Matrix = flash.geom.Matrix;
		var BitmapData = flash.display.BitmapData;
		var Rectangle = flash.geom.Rectangle;
		
		// resolve passed parameters
		var hasPattern = ( pattern_mc instanceof MovieClip );
		if( !hasPattern )		throw new Error( "makeMovieclipSurface() did not receive a pattern MovieClip" );
		
		// internal parameters
		var edgeOpaque = true;
		
		// Create the render-display image.
		// figure out the maximum safe crop area,  so that no out-of-image areas will ever be seen
		var aspectRatio = pattern_mc._width / pattern_mc._height;
		//var rad_45 = 0.78539816339744830961566084581988;		// 45 degrees, in radians  (where the rectangle touches the safe-circle of a square image)
		//var x_scalar = Math.cos(rad_45);
		//var y_scalar = Math.sin(rad_45);	// omitted because, at 45 degrees, sin() and cos() have the same value
		var x_scalar = 0.707106781186548;		// hard-code the resulting value because it will never change
		var y_scalar = x_scalar * aspectRatio;		// x_scalar is used because, at 45 degrees, sin() and cos() have the same value
		var crop_x = Math.round((pattern_mc._width/4)		*(1-x_scalar));			// left offset	(half of a single image  (exclude duplicate buffer)
		var crop_y = Math.round((pattern_mc._height/4)	*(1-y_scalar));			// top offset		(half of a single image  (exclude duplicate buffer)
		var crop_width = Math.round((pattern_mc._width/4		*x_scalar) *2);
		var crop_height = Math.round((pattern_mc._height/4	*y_scalar) *2);
		var crop_mat = new Matrix( 1,0,0,1, -crop_x, -crop_y );
		var crop_pic = new BitmapData(crop_width, crop_height, !edgeOpaque, 0);
		var clipRect = new Rectangle( 0,0, crop_width,crop_height  );
		
		var center_mat = 			new Matrix( 1,0,0,1, -(pattern_mc._width/4), -(pattern_mc._height/4) );		// half width  of a single image iteration  (exclude duplicate buffer on the right)
		var center_neg_mat = 	new Matrix( 1,0,0,1,  (pattern_mc._width/4),  (pattern_mc._height/4) );		// half height of a single image iteration  (exclude duplicate buffer on the bottom)
		
		var output = {
			render_pic: crop_pic, 
			center_mat: center_mat, 
			center_neg_mat: center_neg_mat, 
			crop_mat: crop_mat, 
			clipRect: clipRect
		}
		
		return output;
	}, // makeMovieclipSurface()
	
	

	/*****************************************
	// shrink the view-area
	Resizes the clipping-rectangle and rendering bitmap to the specified size
	renderSurfaceData + newSize_rectangle  =>  new renderSurfaceData with modified  clipRect  +  render_pic
	input:		object			(renderSurface data)
							clipRect
							render_pic
						Rectangle		(newSize_rect)
	output:		object			(new renderSurface data  (shallow-copy if input with modified  clipRect  +  render_pic)
	*****************************************/
	modifyViewSize: function( renderData, newSize_rect ){
		var in_rect = renderData.clipRect;
		var Rectangle = flash.geom.Rectangle;
		var Matrix = flash.geom.Matrix;
		var BitmapData = flash.display.BitmapData;
		if( isNaN(in_rect.width) )					throw new Error("ERROR:  setViewSize()  input_rect does not have a 'width'");
		if( isNaN(in_rect.height) )					throw new Error("ERROR:  setViewSize()  input_rect does not have a 'height'");
		if( isNaN(newSize_rect.width) )			throw new Error("ERROR:  setViewSize()  newSize_rect does not have a 'width'");
		if( isNaN(newSize_rect.height) )		throw new Error("ERROR:  setViewSize()  newSize_rect does not have a 'height'");
		
		var wDiff = in_rect.width - newSize_rect.width;
		var hDiff = in_rect.height - newSize_rect.height;
		if( wDiff < 0 ){
			trace("WARNING: setViewSize()  new width ("+newSize_rect.width+") is bigger than the original clipping-rectangle.  Will use the original size instead.");
			// new width too big => abort
			// return renderData;
		}
		if( hDiff < 0 ){
			trace("WARNING: setViewSize()  new height ("+newSize_rect.height+") is bigger than the original clipping-rectangle.  Will use the original size instead.");
			// new height too big => abort
			// return renderData;
		}
		
		var wHalf = Math.floor( wDiff / 2 );
		var hHalf = Math.floor( hDiff / 2 );
		var output = {};
		for(var nam in renderData){
			output[nam] = renderData[nam];
		}
		output.clipRect = new Rectangle( in_rect.x+wHalf, in_rect.y+hHalf, newSize_rect.width, newSize_rect.height );
		output.render_pic = new BitmapData(  newSize_rect.width, newSize_rect.height,  renderData.render_pic.transparent, 0  );
		return output;
	}, // modifyViewSize()
	
	
	
	/*****************************************
	// apply cropYScale  (change the rotation-point of the output image, by cropping the render-image after rotation)
	Crop the clipping-rectangle and rendering bitmap down to the specified vertical-scale
	renderSurfaceData + yScalar  =>  new renderSurfaceData with modified  clipRect  +  render_pic
	input:		object			(renderSurface data)
							clipRect
							render_pic
						Rectangle		(newSize_rect)
	output:		object			(new renderSurface data  (shallow-copy if input with modified  clipRect  +  render_pic)
	*****************************************/
	applyYCrop: function( renderData, yScale ){
		var in_rect = renderData.clipRect;
		var Rectangle = flash.geom.Rectangle;
		var BitmapData = flash.display.BitmapData;
		if( in_rect instanceof Rectangle === false )		throw new Error("ERROR:  applyYCrop()  in_rect is not a Rectangle");
		if( isNaN(yScale) )		throw new Error("ERROR:  applyYCrop()  yScale is not a Number.");
		if( yScale > 1 )			throw new Error("ERROR:  applyYCrop()  yScale is too big.  It's bigger than 1.");
		if( yScale <= 0 )			throw new Error("ERROR:  applyYCrop()  yScale is too small.  It must be bigger than 0.");
		
		var output = {};
		for(var nam in renderData){
			output[nam] = renderData[nam];
		}
		var newHeight = in_rect.height * yScale;
		output.clipRect = new Rectangle( in_rect.x,in_rect.y, in_rect.width, newHeight );
		output.render_pic = new BitmapData(  in_rect.width, newHeight,  renderData.render_pic.transparent, 0  );
		return output;
	}, // applyYCrop()
	
	
	
	/*****************************************
	coords + pattern = output pic
	input:	bitmap			(pattern bitmap)
					x,y					(view-point top-left coords)
					radian			(view-point angle)
					renderSurfaceData
						render_pic				(BitmapData)		(output bitmap scaled to the maximum size that avoids visible clipping at 45-degree angles on a looping pattern)
						center_mat				(Matrix)				(coords stored as a matrix which controls the rotation-point)
						center_neg_mat		(Matrix)				(a matrix to undo whatever center_mat does)
						crop_mat					(Matrix)				(used to offset the rendered bitmap into the cropped area)
						clipRect					(Rectangle)			(used to crop the rendered bitmap)
					viewScale						(Scalar)				(zoom-in / zoom-out on the rotation-point)
					eraseColor					(Number)				(the color used for the edges of the rendered image when wrapping is disabled)
	output:	bitmap
	*****************************************/
	render: function( pattern_pic, pos_topLeft_p, radian, renderSurfaceData, zoomScale, eraseColor ){
		var Matrix = flash.geom.Matrix;
		var BitmapData = flash.display.BitmapData;
		var Rectangle = flash.geom.Rectangle;
		
		// resolve passed parameters
		if( isNaN(zoomScale) )		var zoomScale = 1;
		var hasPattern = ( pattern_pic instanceof BitmapData )  ||  ( pattern_pic instanceof MovieClip );
		if( !hasPattern )		throw new Error( "render() did not receive a pattern that is BitmapData or MovieClip" );
		var hasCoords = !( pos_topLeft_p.x === undefined  ||  pos_topLeft_p.y === undefined );
		if( !hasCoords )		throw new Error("render() did not receive position coordinates");
		var hasNumCoords = !( isNaN(pos_topLeft_p.x)  ||  isNaN(pos_topLeft_p.y) );
		if( !hasNumCoords )		throw new Error("render() received invalid non-numeric position coordinates");
		var hasRadian = !isNaN( radian );
		if( !hasRadian )			throw new Error("render()  The radian angle is either missing or invalid");
		
		if( !renderSurfaceData )		throw new Error( "render()  renderSurface data is missing" );
		var render_pic = renderSurfaceData.render_pic;
		var center_mat = renderSurfaceData.center_mat;
		var center_neg_mat = renderSurfaceData.center_neg_mat;
		var crop_mat = renderSurfaceData.crop_mat;
		var clipRect = renderSurfaceData.clipRect;
		var has_render_pic = (render_pic instanceof BitmapData);
		if( !has_render_pic )		throw new Error( "render()  renderSurface:  is missing pattern BitmapData" );
		var has_center_mat = (center_mat instanceof Matrix);
		if( !has_center_mat )		throw new Error( "render()  renderSurface:  is missing a center-offset Matrix" );
		var has_center_neg_mat = (center_neg_mat instanceof Matrix);
		if( !has_center_neg_mat )		throw new Error( "render()  renderSurface:  is missing a negative center-offset Matrix" );
		var has_crop_mat = (crop_mat instanceof Matrix);
		if( !has_crop_mat )		throw new Error( "render()  renderSurface:  is missing a crop-offset Matrix" );
		var has_clipRect = (clipRect instanceof Rectangle);
		if( !has_clipRect )		throw new Error( "render()  renderSurface:  is missing a cropping Rectangle" );
		
		// create data
		var position_neg_mat = 		new Matrix( 1,0,0,1, -pos_topLeft_p.x, -pos_topLeft_p.y );
		var angle_mat = 					new Matrix();				angle_mat.rotate( radian );
		
		// apply position and rotation
		var display_mat = new Matrix();
		// display_mat.identity();
		display_mat.concat( center_mat );				// start at -5,-5		(to rotate around the center)
		display_mat.concat( position_neg_mat );	// add the position coords
		display_mat.concat( angle_mat );				// add the angle		(to rotate)
		display_mat.scale( zoomScale, zoomScale );
			var center_neg_mat_scaled = center_neg_mat.clone();
			center_neg_mat_scaled.scale( zoomScale, zoomScale );
		display_mat.concat( center_neg_mat );		// shift +5,+5			(to display the center at the center)
		
		display_mat.concat( crop_mat );					// offset to top-left of crop area
		
		if( eraseColor !== undefined ){
			var full_rect = new Rectangle( 0,0, render_pic.width,render_pic.height );
			render_pic.fillRect( full_rect, eraseColor );
		}
		
		// use "clipRect" to reduce drawing work
		// // If clipRect has an offset (such as when it's smaller than the maximum width or height),  Then position the rendered result in the top-left corner of the output-bitmap.
		if( clipRect.x > 0 || clipRect.y > 0 ){
			display_mat.translate( -clipRect.x, -clipRect.y );
			var clipRect = new Rectangle( 0,0, clipRect.width, clipRect.height );
		}
		// render_pic.draw( pattern_pic, display_mat );		// the size of render_pic will automatically crop down to the correct width and height
		var interpolation = true;
		render_pic.draw( pattern_pic, display_mat, null, null, clipRect, interpolation );		// the size of render_pic will automatically crop down to the correct width and height
		
		return render_pic;
	}, // render()
	
	
	
	null: null
}// loopImageHelper {}